/* =================================================================================================
// 🚀 FILE: assets/js/mega-menu.js
// PURPOSE: Handles Super Mega Menu functionality with dynamic content loading and interactive features.
// ENSURES: Responsive, flexbox-based design with robust debugging and error handling.
// DEPENDENCIES: jQuery, jQuery UI (for draggable/resizable), and Font Awesome (for icons).
// ================================================================================================= */

document.addEventListener('DOMContentLoaded', function () {
    console.log('Super Mega Menu: Initializing modular menu system.');

    // Global variable to track if wellness content is loaded
    let wellnessContentLoaded = false;

    // ============================================================
    // 🧩 Section: Dynamic Content Loader
    // Purpose: Fetches HTML from a partial file and injects it into a target container.
    // ============================================================
    async function loadMenuContent(menuId, filePath) {
        console.log(`Super Mega Menu: Loading content for #${menuId} from ${filePath}`);
        try {
            const response = await fetch(filePath);
            if (!response.ok) {
                throw new Error(`Network response failed: ${response.statusText} for ${filePath}`);
            }
            const content = await response.text();
            const targetElement = document.getElementById(menuId);

            if (!targetElement) {
                throw new Error(`Target element #${menuId} not found in the DOM.`);
            }

            targetElement.innerHTML = content;
            console.log(`Super Mega Menu: Successfully injected content for #${menuId}.`);
            return true;
        } catch (error) {
            console.error(`Super Mega Menu: Failed to load content for #${menuId}.`, error);
            throw error;
        }
    }

    // ============================================================
    // 🌿 WELLNESS DYNAMIC LOADING FUNCTION
    // Purpose: Load wellness mega menu content dynamically
    // ============================================================
    async function loadWellnessContent() {
        console.log('🌿 WELLNESS LOADING: Dynamic loading initiated...');
        
        const loadingIndicator = document.getElementById('wellness-loading');
        const dynamicContainer = document.getElementById('wellness-dynamic-content');
        
        // Show loading indicator
        if (loadingIndicator) {
            loadingIndicator.style.display = 'flex';
            console.log('⏳ WELLNESS LOADING: Loading indicator displayed');
        }
        
        // Check if content is already loaded
        if (wellnessContentLoaded) {
            console.log('✅ WELLNESS CACHE: Content already loaded, skipping fetch');
            hideLoadingIndicator();
            return;
        }
        
        const performanceTracker = trackWellnessPerformance();
        try {
            console.log('📡 WELLNESS FETCH: Requesting partials/_wellness-menu-content.html');
            const response = await fetch('partials/_wellness-menu-content.html');
            
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            
            const wellnessHTML = await response.text();
            console.log('✅ WELLNESS FETCH: Content retrieved successfully');
            console.log('📊 WELLNESS SIZE:', wellnessHTML.length, 'characters loaded');
            
            // Inject content into dynamic container
            if (dynamicContainer) {
                dynamicContainer.innerHTML = wellnessHTML;
                dynamicContainer.style.border = 'none';
                dynamicContainer.style.background = 'transparent';
                dynamicContainer.style.padding = '0';
                console.log('🎯 WELLNESS INJECT: Content injected into dynamic container');
            }
            
            // Mark as loaded
            wellnessContentLoaded = true;
            
            // Hide loading indicator
            setTimeout(() => {
                hideLoadingIndicator();
                console.log('🚀 WELLNESS COMPLETE: Dynamic loading completed');
                performanceTracker.end();
            }, 500);
            
            // Initialize wellness-specific functionality
            initializeWellnessFeatures();
            
        } catch (error) {
            console.error('❌ WELLNESS ERROR: Failed to load wellness content:', error);
            
            if (dynamicContainer) {
                dynamicContainer.innerHTML = `
                    <div class="wellness-error-message">
                        <i class="fas fa-exclamation-triangle"></i>
                        <h4>Unable to Load Wellness Content</h4>
                        <p>Please try refreshing the page or contact support.</p>
                        <button onclick="loadWellnessContent()" class="retry-btn">
                            <i class="fas fa-redo"></i> Try Again
                        </button>
                    </div>
                `;
            }
            
            hideLoadingIndicator();
            performanceTracker.end();
        }
    }

    function hideLoadingIndicator() {
        const loadingIndicator = document.getElementById('wellness-loading');
        if (loadingIndicator) {
            loadingIndicator.style.display = 'none';
            console.log('✅ WELLNESS UI: Loading indicator hidden');
        }
    }

    function initializeWellnessFeatures() {
        console.log('🔧 WELLNESS INIT: Initializing wellness-specific features...');
        
        const wellnessItems = document.querySelectorAll('.wellness-item');
        wellnessItems.forEach(item => {
            item.addEventListener('click', function() {
                console.log(`🛒 WELLNESS DYNAMIC CLICK: ${item.textContent.trim()} selected`);
            });
        });
        
        const wellnessCTAs = document.querySelectorAll('.wellness-cta-btn');
        wellnessCTAs.forEach(btn => {
            btn.addEventListener('click', function() {
                console.log(`🎯 WELLNESS DYNAMIC CTA: ${btn.textContent.trim()} clicked`);
            });
        });
        
        const wellnessCards = document.querySelectorAll('.wellness-category-card');
        wellnessCards.forEach(card => {
            card.addEventListener('mouseenter', function() {
                console.log('🖱️ WELLNESS HOVER: Category card hovered');
            });
        });
        
        console.log('✅ WELLNESS INIT: All wellness features initialized');
    }

    function trackWellnessPerformance() {
        const startTime = performance.now();
        return {
            end: function() {
                const endTime = performance.now();
                const loadTime = (endTime - startTime).toFixed(2);
                console.log(`⏱️ WELLNESS PERFORMANCE: Content loaded in ${loadTime}ms`);
                return loadTime;
            }
        };
    }

    // Inject wellness error styles
    if (!document.getElementById('wellness-error-styles')) {
        const styleElement = document.createElement('style');
        styleElement.id = 'wellness-error-styles';
        styleElement.innerHTML = `
            .wellness-error-message {
                text-align: center;
                padding: 40px 20px;
                background: #fff3cd;
                border: 1px solid #ffeaa7;
                border-radius: 10px;
                color: #856404;
            }
            .wellness-error-message i {
                font-size: 48px;
                color: #f39c12;
                margin-bottom: 15px;
            }
            .wellness-error-message h4 {
                color: #856404;
                font-size: 18px;
                margin-bottom: 10px;
            }
            .wellness-error-message p {
                color: #856404;
                margin-bottom: 20px;
            }
            .retry-btn {
                background: #4baf47;
                color: white;
                border: none;
                padding: 10px 20px;
                border-radius: 5px;
                cursor: pointer;
                font-weight: 600;
                transition: background 0.3s ease;
            }
            .retry-btn:hover {
                background: #12410d;
            }
        `;
        document.head.appendChild(styleElement);
    }

    // ============================================================
    // 🧩 Section: Core Menu Initializer Function
    // Purpose: Initializes all interactive menu features after content is loaded.
    // ============================================================
    function initializeAllInteractiveMenus() {
        console.log('Super Mega Menu: Initializing interactive menu elements...');

        // ============================================================
        // 🧩 Section: Coordinate Logger and Position Memory
        // ============================================================
        const defaultMenuSettings = {
            'vertical-mega-menu-home': { left: -96, top: 78, width: 333, height: 521 },
            'horizontal-mega-menu-pages': { left: -356, top: 64, width: 869, height: 327 },
            'services-mega-menu': { left: -450, top: 64, width: 869, height: 327 }
        };
        let menuPositions = JSON.parse(JSON.stringify(defaultMenuSettings));
        const savedPositions = localStorage.getItem('megaMenuPositions');
        if (savedPositions) {
            try {
                const parsed = JSON.parse(savedPositions);
                if (parsed && typeof parsed === 'object') {
                    for (const menuId in parsed) {
                        if (menuPositions[menuId]) {
                            menuPositions[menuId] = { ...menuPositions[menuId], ...parsed[menuId] };
                        }
                    }
                    console.log('Mega Menu: Loaded saved positions:', menuPositions);
                }
            } catch (error) {
                console.error('Mega Menu: Failed to parse saved positions:', error);
            }
        }

        function logFinalPosition(menuId, left, top, width, height) {
            if (!menuPositions[menuId]) menuPositions[menuId] = {};
            menuPositions[menuId] = { left, top, width, height };
            console.log(`Mega Menu: Saved position for ${menuId} - left: ${left}px, top: ${top}px, width: ${width}px, height: ${height}px`);
            try {
                localStorage.setItem('megaMenuPositions', JSON.stringify(menuPositions));
            } catch (error) {
                console.error('Mega Menu: Failed to save positions to localStorage:', error);
            }
        }

        // ============================================================
        // 🧩 Section: Fix Sidebar Height
        // ============================================================
        function fixSidebarHeight() {
            const sidebar = document.querySelector('#targetElement');
            if (!sidebar) {
                console.warn('Mega Menu: Sidebar (#targetElement) not found');
                return;
            }
            Object.assign(sidebar.style, {
                position: 'fixed',
                top: '0',
                right: '0',
                bottom: '0',
                height: '100vh',
                maxHeight: '100vh',
                width: window.innerWidth <= 768 ? '80%' : '400px',
                maxWidth: '400px',
                backgroundColor: '#ffffff',
                color: '#333333',
                zIndex: '9999',
                overflowY: 'auto',
                boxShadow: '-2px 0 10px rgba(0, 0, 0, 0.2)',
                transition: 'transform 0.3s ease-in-out',
                padding: '20px',
                display: sidebar.classList.contains('side_bar_hidden') ? 'none' : 'flex',
                flexDirection: 'column'
            });
            sidebar.style.transform = sidebar.classList.contains('slideInRight') ? 'translateX(0)' : 'translateX(100%)';
            console.log('Mega Menu: Sidebar styles applied');
        }

        // ============================================================
        // 🧩 Section: Fix Mobile Menu Visibility
        // ============================================================
        function fixMobileMenuVisibility() {
            const accordionToggles = document.querySelectorAll('#targetElement .accordion-toggle, .mobile-menu .accordion-toggle');
            accordionToggles.forEach(toggle => {
                Object.assign(toggle.style, {
                    backgroundColor: '#407a5a',
                    color: '#ffffff',
                    padding: '10px 15px',
                    margin: '5px 0',
                    borderRadius: '4px',
                    display: 'block',
                    visibility: 'visible',
                    opacity: '1',
                    cursor: 'pointer',
                    fontWeight: 'bold',
                    textAlign: 'left'
                });
            });

            const accordionContents = document.querySelectorAll('#targetElement .accordion-content, .mobile-menu .accordion-content');
            accordionContents.forEach(content => {
                Object.assign(content.style, {
                    backgroundColor: '#f2f2f2',
                    color: '#333333',
                    padding: '10px',
                    margin: '0 0 10px 0',
                    borderRadius: '0 0 4px 4px',
                    borderLeft: '3px solid #407a5a',
                    display: content.classList.contains('active') ? 'block' : 'none'
                });
            });

            const textElements = document.querySelectorAll('#targetElement *, .mobile-menu *');
            textElements.forEach(el => {
                if (el.classList.contains('accordion-toggle') || el.classList.contains('accordion-content')) return;
                el.style.color = '#333333';
                el.style.textShadow = 'none';
            });

            const links = document.querySelectorAll('#targetElement a, .mobile-menu a');
            links.forEach(link => {
                Object.assign(link.style, {
                    color: '#407a5a',
                    textDecoration: 'none',
                    display: 'block',
                    padding: '5px 0',
                    margin: '2px 0',
                    visibility: 'visible',
                    opacity: '1'
                });
            });

            const headings = document.querySelectorAll('#targetElement h3, #targetElement h4, #targetElement h5, #targetElement h6, .mobile-menu h3, .mobile-menu h4, .mobile-menu h5, .mobile-menu h6');
            headings.forEach(heading => {
                Object.assign(heading.style, {
                    color: '#407a5a',
                    fontWeight: 'bold',
                    margin: '10px 0 5px 0',
                    visibility: 'visible',
                    opacity: '1'
                });
            });

            console.log('Mega Menu: Mobile menu visibility styles applied');
        }

        // ============================================================
        // 🧩 Section: Fix Mega Menu Dropdowns
        // ============================================================
        function fixMegaMenuDropdowns() {
            const menus = [
                { id: 'vertical-mega-menu-home', selector: '#vertical-mega-menu-home' },
                { id: 'horizontal-mega-menu-pages', selector: '#horizontal-mega-menu-pages' }
            ];
            menus.forEach(({ id, selector }) => {
                const menu = document.querySelector(selector);
                if (!menu) {
                    console.warn(`Mega Menu: ${id} not found`);
                    return;
                }
                Object.assign(menu.style, {
                    backgroundColor: '#ffffff',
                    color: '#333333',
                    zIndex: '20001'
                });
                menu.querySelectorAll('*').forEach(el => {
                    if (!el.style) return;
                    el.style.color = '#333333';
                    el.style.textShadow = 'none';
                });
                menu.querySelectorAll('a').forEach(link => {
                    Object.assign(link.style, {
                        color: '#407a5a',
                        textDecoration: 'none',
                        display: 'block',
                        padding: '5px 0',
                        margin: '2px 0',
                        visibility: 'visible',
                        opacity: '1'
                    });
                });
                console.log(`Mega Menu: Styles applied to ${id}`);
            });
        }

        // ============================================================
        // 🧩 Section: Menu Initialization
        // ============================================================
        const menuRefs = {
            verticalMenu: getMenuRef('#vertical-mega-menu-home'),
            horizontalMenu: getMenuRef('#horizontal-mega-menu-pages'),
            sidebar: getMenuRef('#targetElement')
        };

        function getMenuRef(selector) {
            const elements = document.querySelectorAll(selector);
            if (elements.length > 1) {
                console.warn(`Mega Menu: Multiple elements for ${selector}, using first`);
                const primaryElement = elements[0];
                Array.from(elements).slice(1).forEach(element => {
                    Array.from(element.children).forEach(child => {
                        if (child && !primaryElement.querySelector(`[data-id="${child.getAttribute('data-id')}"]`)) {
                            primaryElement.appendChild(child.cloneNode(true));
                        }
                    });
                    element.remove();
                });
                return { current: primaryElement };
            }
            return { current: elements[0] || null };
        }

        Object.entries(menuRefs).forEach(([key, ref]) => {
            if (ref.current) {
                console.log(`Mega Menu: Initialized ${key} (#${ref.current.id})`);
                ref.current.style.display = 'none';
                ref.current.style.backgroundColor = '#ffffff';
                ref.current.style.zIndex = '20001';
            } else {
                console.warn(`Mega Menu: ${key} not found`);
            }
        });

        // ============================================================
        // 🚀 Section: Global Tab Function
        // ============================================================
        window.showTab = function(tabId) {
            console.log(`Super Mega Menu: Switching to tab: ${tabId}`);
            const allTabContents = document.querySelectorAll('.tab-content');
            allTabContents.forEach(content => {
                content.classList.remove('active');
                content.style.display = 'none';
            });

            const allTabLinks = document.querySelectorAll('.tab-link');
            allTabLinks.forEach(link => link.classList.remove('active'));

            const selectedTab = document.getElementById(tabId);
            if (selectedTab) {
                selectedTab.classList.add('active');
                selectedTab.style.display = 'block';
                console.log(`Super Mega Menu: Activated tab: ${tabId}`);
            } else {
                console.warn(`Super Mega Menu: Tab ${tabId} not found`);
            }

            const clickedTabLink = document.querySelector(`[data-tab="${tabId}"]`);
            if (clickedTabLink) {
                clickedTabLink.classList.add('active');
            }

            // Load wellness content if wellness tab is selected
            if (tabId === 'wellness-tab') {
                loadWellnessContent();
            }
        };

        // ============================================================
        // 🚀 Section: Tabbed Menu Logic
        // ============================================================
        const tabLinks = document.querySelectorAll('.tab-link');
        tabLinks.forEach(link => {
            link.addEventListener('click', function(event) {
                event.preventDefault();
                event.stopPropagation();
                const tabId = this.getAttribute('data-tab');
                if (tabId) {
                    window.showTab(tabId);
                }
            });
        });

        // Initialize first tab
        setTimeout(() => {
            if (!document.querySelector('.tab-content.active')) {
                const firstTab = document.querySelector('.tab-content');
                if (firstTab) {
                    window.showTab(firstTab.id);
                }
            }
        }, 100);

        // ============================================================
        // 🧩 Section: Accordion Toggles
        // ============================================================
        function initializeAccordions(selector, logPrefix) {
            const toggles = document.querySelectorAll(selector);
            if (!toggles.length) {
                console.warn(`${logPrefix}: No toggles found`);
                return;
            }
            console.log(`${logPrefix}: Found ${toggles.length} toggles`);
            toggles.forEach(toggle => {
                toggle.addEventListener('click', function(e) {
                    e.preventDefault();
                    const content = this.nextElementSibling;
                    const isExpanded = this.getAttribute('aria-expanded') === 'true';
                    toggles.forEach(other => {
                        if (other !== toggle) {
                            other.setAttribute('aria-expanded', 'false');
                            other.nextElementSibling.classList.remove('active');
                            other.nextElementSibling.style.display = 'none';
                        }
                    });
                    this.setAttribute('aria-expanded', !isExpanded);
                    content.classList.toggle('active', !isExpanded);
                    content.style.display = isExpanded ? 'none' : 'block';
                    content.style.backgroundColor = '#f2f2f2';
                    console.log(`${logPrefix}: ${isExpanded ? 'Collapsed' : 'Expanded'} ${toggle.textContent.trim()}`);
                });
            });
        }

        initializeAccordions('.accordion-toggle', 'Mega Menu Accordion');
        initializeAccordions('.sub-accordion-toggle', 'Mega Menu Sub-Accordion');

        // ============================================================
        // 🧩 Section: Menu Positioning
        // ============================================================
        function positionMenu(menu, parent) {
            if (!menu || !parent) return;
            console.log(`Mega Menu: Positioning ${menu.id}`);
            const settings = menuPositions[menu.id] || defaultMenuSettings[menu.id] || {
                left: 0,
                top: parent.getBoundingClientRect().bottom + window.scrollY,
                width: menu.id.includes('horizontal') ? 869 : 333,
                height: menu.id.includes('horizontal') ? 327 : 521
            };
            Object.assign(menu.style, {
                position: 'absolute',
                top: `${settings.top}px`,
                left: `${settings.left}px`,
                width: `${settings.width}px`,
                height: `${settings.height}px`,
                zIndex: '20001',
                backgroundColor: '#ffffff',
                boxShadow: '0 4px 8px rgba(0, 0, 0, 0.1)',
                visibility: 'visible',
                opacity: '1'
            });
            if (menu.id.includes('horizontal')) {
                menu.style.display = 'flex';
                menu.style.flexWrap = 'wrap';
                menu.style.justifyContent = 'space-between';
                const arrow = menu.querySelector('.arrow-indicator');
                if (arrow) {
                    const parentRect = parent.getBoundingClientRect();
                    arrow.style.left = `${parentRect.left + parentRect.width / 2 - parseFloat(menu.style.left)}px`;
                    arrow.style.position = 'absolute';
                    arrow.style.top = '-10px';
                    arrow.style.borderColor = 'transparent transparent #ffffff transparent';
                }
            }
            logFinalPosition(menu.id, settings.left, settings.top, settings.width, settings.height);
        }

        // ============================================================
        // 🚀 Section: Unified Click Handler
        // ============================================================
        let activeMenu = null;
        document.addEventListener('click', function(e) {
            const dropdownTrigger = e.target.closest('.dropdown-hover');
            if (dropdownTrigger) {
                const menu = dropdownTrigger.nextElementSibling;
                if (menu && (menu.classList.contains('mega-menu-container') || menu.classList.contains('sub-menu'))) {
                    e.preventDefault();
                    const isVisible = menu.classList.contains('active');
                    if (activeMenu && activeMenu !== menu) {
                        activeMenu.classList.remove('active');
                        activeMenu.style.display = 'none';
                    }
                    if (isVisible) {
                        menu.classList.remove('active');
                        menu.style.display = 'none';
                        activeMenu = null;
                    } else {
                        menu.classList.add('active');
                        menu.style.display = menu.id.includes('horizontal') ? 'flex' : 'block';
                        positionMenu(menu, dropdownTrigger);
                        activeMenu = menu;
                    }
                    console.log(`Mega Menu: ${isVisible ? 'Closed' : 'Opened'} ${menu.id}`);
                    return;
                }
            }
            if (activeMenu && !activeMenu.contains(e.target)) {
                activeMenu.classList.remove('active');
                activeMenu.style.display = 'none';
                console.log(`Mega Menu: Closed ${activeMenu.id} via outside click`);
                activeMenu = null;
            }
        });

        // ============================================================
        // 🧩 Section: Detachable and Resizable Menu
        // ============================================================
        const detachableMenus = document.querySelectorAll('.detachable');
        detachableMenus.forEach(menu => {
            let isDragging = false, isResizing = false, currentX = 0, currentY = 0, initialX = 0, initialY = 0;
            if (menuPositions[menu.id]) {
                const pos = menuPositions[menu.id];
                currentX = pos.left;
                currentY = pos.top;
                Object.assign(menu.style, {
                    position: 'absolute',
                    left: `${pos.left}px`,
                    top: `${pos.top}px`,
                    width: `${pos.width}px`,
                    height: `${pos.height}px`
                });
            }

            menu.addEventListener('mousedown', function(e) {
                if (e.target.classList.contains('mega-menu-container') || e.target.classList.contains('super-mega-menu-wrapper')) {
                    isDragging = true;
                    initialX = e.clientX - currentX;
                    initialY = e.clientY - currentY;
                    menu.style.cursor = 'grabbing';
                }
            });

            document.addEventListener('mousemove', function(e) {
                if (isDragging) {
                    e.preventDefault();
                    currentX = e.clientX - initialX;
                    currentY = e.clientY - initialY;
                    menu.style.left = `${currentX}px`;
                    menu.style.top = `${currentY}px`;
                }
            });

            document.addEventListener('mouseup', function() {
                if (isDragging) {
                    isDragging = false;
                    menu.style.cursor = 'default';
                    logFinalPosition(menu.id, currentX, currentY, parseInt(menu.style.width), parseInt(menu.style.height));
                }
            });

            const handles = [
                { class: 'resize-top', style: 'top: -5px; left: 0; right: 0; height: 10px; cursor: ns-resize;' },
                { class: 'resize-bottom', style: 'bottom: -5px; left: 0; right: 0; height: 10px; cursor: ns-resize;' },
                { class: 'resize-left', style: 'left: -5px; top: 0; bottom: 0; width: 10px; cursor: ew-resize;' },
                { class: 'resize-right', style: 'right: -5px; top: 0; bottom: 0; width: 10px; cursor: ew-resize;' },
                { class: 'resize-top-left', style: 'top: -5px; left: -5px; width: 15px; height: 15px; cursor: nwse-resize;' },
                { class: 'resize-top-right', style: 'top: -5px; right: -5px; width: 15px; height: 15px; cursor: nesw-resize;' },
                { class: 'resize-bottom-left', style: 'bottom: -5px; left: -5px; width: 15px; height: 15px; cursor: nesw-resize;' },
                { class: 'resize-bottom-right', style: 'bottom: -5px; right: -5px; width: 15px; height: 15px; cursor: nwse-resize;' }
            ];

            menu.querySelectorAll('.resize-handle').forEach(h => h.remove());
            handles.forEach(h => {
                const handle = document.createElement('div');
                handle.className = `resize-handle ${h.class}`;
                handle.style.cssText = `position: absolute; background: rgba(64, 122, 90, 0.7); z-index: 20002; ${h.style}`;
                handle.addEventListener('mouseenter', function() { this.style.background = 'rgba(64, 122, 90, 1)'; });
                handle.addEventListener('mouseleave', function() { this.style.background = 'rgba(64, 122, 90, 0.7)'; });
                menu.appendChild(handle);
            });

            menu.addEventListener('mousedown', function(e) {
                if (e.target.classList.contains('resize-handle')) {
                    isResizing = true;
                    const startX = e.clientX;
                    const startY = e.clientY;
                    const startWidth = parseInt(getComputedStyle(menu).width, 10);
                    const startHeight = parseInt(getComputedStyle(menu).height, 10);
                    currentX = parseInt(menu.style.left || '0', 10);
                    currentY = parseInt(menu.style.top || '0', 10);
                    const direction = e.target.className;
                    e.preventDefault();
                    e.stopPropagation();

                    document.addEventListener('mousemove', function resize(e) {
                        if (!isResizing) return;
                        e.preventDefault();
                        let newWidth = startWidth, newHeight = startHeight, newLeft = currentX, newTop = currentY;
                        if (direction.includes('right')) newWidth = Math.max(300, startWidth + (e.clientX - startX));
                        if (direction.includes('left')) {
                            const wc = startX - e.clientX;
                            newWidth = Math.max(300, startWidth + wc);
                            newLeft = currentX - wc;
                        }
                        if (direction.includes('bottom')) newHeight = Math.max(200, startHeight + (e.clientY - startY));
                        if (direction.includes('top')) {
                            const hc = startY - e.clientY;
                            newHeight = Math.max(200, startHeight + hc);
                            newTop = currentY - hc;
                        }
                        menu.style.width = `${newWidth}px`;
                        menu.style.height = `${newHeight}px`;
                        menu.style.left = `${newLeft}px`;
                        menu.style.top = `${newTop}px`;
                    });

                    document.addEventListener('mouseup', function stopResize() {
                        if (isResizing) {
                            isResizing = false;
                            currentX = parseInt(menu.style.left || '0', 10);
                            currentY = parseInt(menu.style.top || '0', 10);
                            logFinalPosition(menu.id, currentX, currentY, parseInt(menu.style.width), parseInt(menu.style.height));
                            document.removeEventListener('mousemove', resize);
                            document.removeEventListener('mouseup', stopResize);
                        }
                    }, { once: true });
                }
            });
        });

        // ============================================================
        // 🧩 Section: Mobile Menu
        // ============================================================
        const openButton = document.querySelector('#openButton');
        const closeButton = document.querySelector('#closeButton');
        const sidebar = menuRefs.sidebar.current;
        if (openButton && closeButton && sidebar) {
            openButton.addEventListener('click', function() {
                sidebar.classList.remove('side_bar_hidden');
                sidebar.classList.add('slideInRight');
                fixSidebarHeight();
                console.log('Mobile Menu: Sidebar opened');
            });
            closeButton.addEventListener('click', function() {
                sidebar.classList.add('side_bar_hidden');
                sidebar.classList.remove('slideInRight');
                console.log('Mobile Menu: Sidebar closed');
            });
            window.addEventListener('resize', fixSidebarHeight);
        } else {
            console.warn('Mobile Menu: Missing sidebar elements');
        }

        // ============================================================
        // 🧩 Section: Final Fixes and Listeners
        // ============================================================
        fixSidebarHeight();
        fixMobileMenuVisibility();
        fixMegaMenuDropdowns();
        window.addEventListener('resize', () => {
            fixSidebarHeight();
            fixMobileMenuVisibility();
            fixMegaMenuDropdowns();
        });

        function clearStuckMenus() {
            document.querySelectorAll('.mega-menu-container').forEach(menu => {
                menu.style.display = 'none';
                menu.style.visibility = 'hidden';
                menu.style.opacity = '0';
            });
            if (sidebar) {
                sidebar.classList.add('side_bar_hidden');
                sidebar.classList.remove('slideInRight');
            }
            console.log('Mega Menu: Cleared stuck menus');
        }
        clearStuckMenus();

        // ============================================================
        // 🧩 Section: jQuery-Based Enhancements
        // ============================================================
        if (typeof jQuery !== 'undefined' && jQuery.fn.draggable && jQuery.fn.resizable) {
            jQuery('.mega-menu-container').each(function() {
                const $container = jQuery(this);
                const menuId = $container.attr('id');
                $container.resizable({
                    handles: 'all',
                    minWidth: 200,
                    minHeight: 100,
                    start: () => console.log(`Mega Menu (jQuery): Resizing ${menuId}`),
                    stop: (event, ui) => {
                        logFinalPosition(menuId, ui.position.left, ui.position.top, ui.size.width, ui.size.height);
                    }
                }).draggable({
                    handle: '.mega-menu-header',
                    containment: 'window',
                    start: () => console.log(`Mega Menu (jQuery): Dragging ${menuId}`),
                    stop: (event, ui) => {
                        logFinalPosition(menuId, ui.position.left, ui.position.top, $container.width(), $container.height());
                    }
                });
                const savedPosition = menuPositions[menuId];
                if (savedPosition) {
                    $container.css({
                        left: savedPosition.left,
                        top: savedPosition.top,
                        width: savedPosition.width,
                        height: savedPosition.height
                    });
                }
            });

            jQuery('.mega-menu-toggle').on('click', function() {
                const targetId = jQuery(this).data('target');
                jQuery(`#${targetId}`).toggleClass('visible');
                console.log(`Mega Menu (jQuery): Toggled #${targetId}`);
            });

            jQuery('.mega-menu-close').on('click', function() {
                jQuery(this).closest('.mega-menu-container').removeClass('visible');
                console.log('Mega Menu (jQuery): Closed menu');
            });
        } else {
            console.warn('Mega Menu: jQuery or jQuery UI not available, skipping enhancements');
        }
    }

    // ============================================================
    // 🌿 GLOBAL WELLNESS FUNCTION (Exposed to window for onclick events)
    // Purpose: Make loadWellnessContent available globally for HTML onclick
    // ============================================================
    window.loadWellnessContent = loadWellnessContent;

    // ============================================================
    // 🚀 Section: Main Execution Block
    // ============================================================
    async function main() {
        try {
            await Promise.all([
                loadMenuContent('vertical-mega-menu-home', 'partials/_home-menu-content.html'),
                loadMenuContent('horizontal-mega-menu-pages', 'partials/_horizontal-menu-content.html'),
                loadMenuContent('services-mega-menu', 'partials/_services-menu-content.html')
            ]);
            console.log('Mega Menu: All partials loaded');
            initializeAllInteractiveMenus();
        } catch (error) {
            console.error('Mega Menu: Critical error during loading:', error);
        }
    }

    main();
});

// ============================================================
// 🧩 Section: Export Default
// ============================================================
export default {};